/*
 * Decompiled with CFR 0.152.
 */
package mage.server;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import mage.cards.repository.CardInfo;
import mage.cards.repository.CardRepository;
import mage.game.Game;
import mage.game.Table;
import mage.game.tournament.Tournament;
import mage.server.ChatSession;
import mage.server.DisconnectReason;
import mage.server.User;
import mage.server.game.GameController;
import mage.server.managers.ChatManager;
import mage.server.managers.ManagerFactory;
import mage.util.GameLog;
import mage.utils.SystemUtil;
import mage.view.ChatMessage;
import org.apache.log4j.Logger;

public class ChatManagerImpl
implements ChatManager {
    private static final Logger logger = Logger.getLogger(ChatManagerImpl.class);
    private static final HashMap<String, String> lastUserMessages = new HashMap();
    private final ManagerFactory managerFactory;
    private final ConcurrentHashMap<UUID, ChatSession> chatSessions = new ConcurrentHashMap();
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    final Pattern cardNamePattern = Pattern.compile("\\[\\[(.*?)\\]\\]");
    private static final String COMMANDS_LIST = "<br/>List of commands:<br/>\\history or \\h [username] - shows the history of a player<br/>\\me - shows the history of the current player<br/>\\list or \\l - Show a list of commands<br/>\\whisper or \\w [player name] [text] - whisper to the player with the given name<br/>\\card Card Name - Print oracle text for card<br/>[Card Name] - Show a highlighted card name<br/>\\ignore - shows your ignore list on this server.<br/>\\ignore [username] - add username to ignore list (they won't be able to chat or join to your game).<br/>\\unignore [username] - remove a username from your ignore list on this server.";
    final Pattern getCardTextPattern = Pattern.compile("^.card *(.*)");

    public ChatManagerImpl(ManagerFactory managerFactory) {
        this.managerFactory = managerFactory;
    }

    @Override
    public UUID createRoomChatSession(UUID roomId) {
        return this.createChatSession("Room " + roomId).withRoom(roomId).getChatId();
    }

    @Override
    public UUID createTourneyChatSession(Tournament tournament) {
        return this.createChatSession("Tourney " + tournament.getId()).withTourney(tournament).getChatId();
    }

    @Override
    public UUID createTableChatSession(Table table) {
        return this.createChatSession("Table " + table.getId()).withTable(table).getChatId();
    }

    @Override
    public UUID createGameChatSession(Game game) {
        return this.createChatSession("Game " + game.getId()).withGame(game).getChatId();
    }

    private ChatSession createChatSession(String info) {
        ChatSession chatSession = new ChatSession(this.managerFactory, info);
        this.chatSessions.put(chatSession.getChatId(), chatSession);
        return chatSession;
    }

    @Override
    public void joinChat(UUID chatId, UUID userId) {
        ChatSession chatSession = this.chatSessions.get(chatId);
        if (chatSession != null) {
            chatSession.join(userId);
        } else {
            logger.trace((Object)("Chat to join not found - chatId: " + chatId + " userId: " + userId));
        }
    }

    @Override
    public void leaveChat(UUID chatId, UUID userId) {
        ChatSession chatSession = this.chatSessions.get(chatId);
        if (chatSession != null && chatSession.hasUser(userId, false)) {
            chatSession.disconnectUser(userId, DisconnectReason.DisconnectedByUser);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void destroyChatSession(UUID chatId) {
        ChatSession chatSession;
        if (chatId != null && (chatSession = this.chatSessions.get(chatId)) != null) {
            if (this.chatSessions.containsKey(chatId)) {
                Lock w = this.lock.writeLock();
                w.lock();
                try {
                    this.chatSessions.remove(chatId);
                }
                finally {
                    w.unlock();
                }
                logger.trace((Object)("Chat removed - chatId: " + chatId));
            } else {
                logger.trace((Object)("Chat to destroy does not exist - chatId: " + chatId));
            }
        }
    }

    @Override
    public void broadcast(UUID chatId, String userName, String message, ChatMessage.MessageColor color, boolean withTime, Game game, ChatMessage.MessageType messageType, ChatMessage.SoundToPlay soundToPlay) {
        String finalMessage = message;
        ChatSession chatSession = this.chatSessions.get(chatId);
        Optional<User> user = this.managerFactory.userManager().getUserByName(userName);
        if (chatSession != null) {
            if ((message.startsWith("\\") || message.startsWith("/")) && user.isPresent()) {
                if (!this.performUserCommand(user.get(), message, chatId, false)) {
                    this.performUserCommand(user.get(), message, chatId, true);
                }
                return;
            }
            if (messageType != ChatMessage.MessageType.GAME && !userName.isEmpty() && user.isPresent()) {
                String messageId;
                if (messageType == ChatMessage.MessageType.TALK && user.get().getChatLockedUntil() != null) {
                    if (user.get().getChatLockedUntil().compareTo(Calendar.getInstance().getTime()) > 0) {
                        this.chatSessions.get(chatId).broadcastInfoToUser(user.get(), "Your chat is muted until " + SystemUtil.dateFormat.format(user.get().getChatLockedUntil()));
                        return;
                    }
                    user.get().setChatLockedUntil(null);
                }
                if ((messageId = chatId.toString() + message).equals(lastUserMessages.get(userName))) {
                    this.chatSessions.get(chatId).broadcastInfoToUser(user.get(), "Ignore duplicated message");
                    return;
                }
                lastUserMessages.put(userName, messageId);
                if (message.length() > 500) {
                    message = message.replaceFirst("^(.{500}).*", "$1 (rest of message truncated)");
                }
                String messageToCheck = message;
                Matcher matchPattern = this.cardNamePattern.matcher(message);
                int foundCount = 0;
                while (matchPattern.find() && ++foundCount <= 5) {
                    String searchName = matchPattern.group(1);
                    CardInfo cardInfo = CardRepository.instance.findCard(searchName, true);
                    if (cardInfo == null) continue;
                    String newMessagePart = GameLog.getColoredObjectIdName(cardInfo.createCard().getColor(), UUID.randomUUID(), cardInfo.getName(), "", cardInfo.getName());
                    messageToCheck = messageToCheck.replaceFirst("\\[\\[" + searchName + "\\]\\]", "[[" + newMessagePart + "]]");
                }
                finalMessage = messageToCheck;
            }
            chatSession.broadcast(userName, finalMessage, color, withTime, game, messageType, soundToPlay);
        }
    }

    private boolean performUserCommand(User user, String message, UUID chatId, boolean doError) {
        String rest;
        int first;
        String command = message.substring(1).trim().toUpperCase(Locale.ENGLISH);
        if (doError) {
            message = message + ("<br/>Invalid User Command '" + message + "'." + COMMANDS_LIST);
            message = message + "<br/>Type <font color=green>\\w " + user.getName() + " profanity 0 (or 1 or 2)</font> to use/not use the profanity filter";
            this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
            return true;
        }
        if (command.startsWith("H ") || command.startsWith("HISTORY ")) {
            message = message + "<br/>" + this.managerFactory.userManager().getUserHistory(message.substring(command.startsWith("H ") ? 3 : 9));
            this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
            return true;
        }
        if (command.equals("ME")) {
            message = message + "<br/>" + this.managerFactory.userManager().getUserHistory(user.getName());
            this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
            return true;
        }
        if (command.startsWith("GAME")) {
            String gameId;
            message = message + "<br/>";
            ChatSession session = this.chatSessions.get(chatId);
            if (session != null && session.getInfo() != null && (gameId = session.getInfo()).startsWith("Game ")) {
                UUID id = UUID.fromString(gameId.substring(5));
                for (Map.Entry<UUID, GameController> entry : this.managerFactory.gameManager().getGameController().entrySet()) {
                    GameController controller;
                    if (!entry.getKey().equals(id) || (controller = entry.getValue()) == null) continue;
                    message = message + controller.getGameStateDebugMessage();
                    this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
                }
            }
            return true;
        }
        if (command.startsWith("FIX")) {
            String gameId;
            message = message + "<br/>";
            ChatSession session = this.chatSessions.get(chatId);
            if (session != null && session.getInfo() != null && (gameId = session.getInfo()).startsWith("Game ")) {
                UUID id = UUID.fromString(gameId.substring(5));
                for (Map.Entry<UUID, GameController> entry : this.managerFactory.gameManager().getGameController().entrySet()) {
                    GameController controller;
                    if (!entry.getKey().equals(id) || (controller = entry.getValue()) == null) continue;
                    message = message + controller.attemptToFixGame(user);
                    this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
                }
            }
            return true;
        }
        if (command.equals("PINGS")) {
            String gameId;
            message = message + "<br/>";
            ChatSession session = this.chatSessions.get(chatId);
            if (session != null && session.getInfo() != null && (gameId = session.getInfo()).startsWith("Game ")) {
                UUID id = UUID.fromString(gameId.substring(5));
                for (Map.Entry<UUID, GameController> entry : this.managerFactory.gameManager().getGameController().entrySet()) {
                    GameController controller;
                    if (!entry.getKey().equals(id) || (controller = entry.getValue()) == null) continue;
                    message = message + controller.getPingsInfo();
                    this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
                }
            }
            return true;
        }
        if (command.startsWith("CARD ")) {
            Matcher matchPattern = this.getCardTextPattern.matcher(message.toLowerCase(Locale.ENGLISH));
            if (matchPattern.find()) {
                String cardName = matchPattern.group(1);
                CardInfo cardInfo = CardRepository.instance.findPreferredCoreExpansionCard(cardName);
                if (cardInfo != null) {
                    cardInfo.getRules();
                    message = "<font color=orange>" + cardInfo.getName() + "</font>: Cost:" + cardInfo.getManaCosts(CardInfo.ManaCostSide.ALL).toString() + ",  Types:" + cardInfo.getTypes().toString() + ", ";
                    for (String rule : cardInfo.getRules()) {
                        message = message + rule;
                    }
                } else {
                    message = "Couldn't find: " + cardName;
                }
            }
            this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
            return true;
        }
        if ((command.startsWith("W ") || command.startsWith("WHISPER ")) && (first = (rest = message.substring(command.startsWith("W ") ? 3 : 9)).indexOf(32)) > 1) {
            String userToName = rest.substring(0, first);
            rest = rest.substring(first + 1).trim();
            Optional<User> userTo = this.managerFactory.userManager().getUserByName(userToName);
            if (userTo.isPresent()) {
                if (!this.chatSessions.get(chatId).broadcastWhisperToUser(user, userTo.get(), rest)) {
                    message = message + ("<br/>User " + userToName + " not found");
                    this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
                }
            } else {
                message = message + ("<br/>User " + userToName + " not found");
                this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
            }
            return true;
        }
        if (command.equals("L") || command.equals("LIST")) {
            message = message + COMMANDS_LIST;
            message = message + "<br/>Type <font color=green>\\w " + user.getName() + " profanity 0 (or 1 or 2)</font> to use/not use the profanity filter";
            this.chatSessions.get(chatId).broadcastInfoToUser(user, message);
            return true;
        }
        return false;
    }

    @Override
    public void sendReconnectMessage(UUID userId) {
        this.managerFactory.userManager().getUser(userId).ifPresent(user -> this.getChatSessions().stream().filter(chat -> chat.hasUser(userId, true)).forEach(chatSession -> chatSession.broadcast(null, user.getName() + " has reconnected", ChatMessage.MessageColor.BLUE, true, null, ChatMessage.MessageType.STATUS, null)));
    }

    @Override
    public void sendMessageToUserChats(UUID userId, String message) {
        this.managerFactory.userManager().getUser(userId).ifPresent(user -> {
            List<ChatSession> chatSessions = this.getChatSessions().stream().filter(chat -> !chat.getChatId().equals(this.managerFactory.gamesRoomManager().getMainChatId())).filter(chat -> chat.hasUser(userId, true)).collect(Collectors.toList());
            if (chatSessions.size() > 0) {
                logger.debug((Object)("INFORM OPPONENTS by " + user.getName() + ": " + message));
                chatSessions.forEach(chatSession -> chatSession.broadcast(null, message, ChatMessage.MessageColor.BLUE, true, null, ChatMessage.MessageType.STATUS, null));
            }
        });
    }

    @Override
    public void removeUser(UUID userId, DisconnectReason reason) {
        for (ChatSession chatSession : this.getChatSessions()) {
            if (!chatSession.hasUser(userId, false)) continue;
            chatSession.disconnectUser(userId, reason);
        }
    }

    @Override
    public List<ChatSession> getChatSessions() {
        Lock r = this.lock.readLock();
        r.lock();
        try {
            ArrayList<ChatSession> arrayList = new ArrayList<ChatSession>(this.chatSessions.values());
            return arrayList;
        }
        finally {
            r.unlock();
        }
    }

    private void clearUserMessageStorage() {
        lastUserMessages.clear();
    }

    @Override
    public void checkHealth() {
        this.clearUserMessageStorage();
    }
}

