/*
 * Decompiled with CFR 0.152.
 */
package mage.server;

import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.ConcurrentModificationException;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import mage.MageException;
import mage.cards.decks.DeckCardLists;
import mage.cards.decks.DeckValidatorFactory;
import mage.cards.repository.CardRepository;
import mage.cards.repository.ExpansionRepository;
import mage.collectors.DataCollectorServices;
import mage.constants.ManaType;
import mage.constants.PlayerAction;
import mage.constants.TableState;
import mage.game.Table;
import mage.game.match.MatchOptions;
import mage.game.tournament.TournamentOptions;
import mage.interfaces.Action;
import mage.interfaces.ActionWithResult;
import mage.interfaces.MageServer;
import mage.interfaces.ServerState;
import mage.interfaces.callback.ClientCallback;
import mage.interfaces.callback.ClientCallbackMethod;
import mage.players.PlayerType;
import mage.players.net.UserData;
import mage.remote.MageVersionException;
import mage.server.AuthorizedUser;
import mage.server.AuthorizedUserRepository;
import mage.server.Main;
import mage.server.Session;
import mage.server.TableController;
import mage.server.User;
import mage.server.draft.CubeFactory;
import mage.server.game.GameFactory;
import mage.server.game.GamesRoom;
import mage.server.game.PlayerFactory;
import mage.server.managers.ManagerFactory;
import mage.server.services.impl.FeedbackServiceImpl;
import mage.server.tournament.TournamentFactory;
import mage.server.util.ServerMessagesUtil;
import mage.util.DebugUtil;
import mage.utils.ActionWithBooleanResult;
import mage.utils.ActionWithNullNegativeResult;
import mage.utils.ActionWithTableViewResult;
import mage.utils.CompressUtil;
import mage.utils.MageVersion;
import mage.utils.SystemUtil;
import mage.view.ChatMessage;
import mage.view.DraftPickView;
import mage.view.GameView;
import mage.view.MatchView;
import mage.view.RoomUsersView;
import mage.view.TableView;
import mage.view.TournamentView;
import mage.view.UserView;
import org.apache.log4j.Logger;
import org.unbescape.html.HtmlEscape;

public class MageServerImpl
implements MageServer {
    private static final Logger logger = Logger.getLogger(MageServerImpl.class);
    private final ExecutorService callExecutor;
    private static final SecureRandom RANDOM = new SecureRandom();
    private final ManagerFactory managerFactory;
    private final String adminPassword;
    private final boolean testMode;
    private final boolean detailsMode;
    private final LinkedHashMap<String, String> activeAuthTokens = new LinkedHashMap<String, String>(){

        @Override
        protected boolean removeEldestEntry(Map.Entry<String, String> eldest) {
            return this.size() > 1024;
        }
    };

    public MageServerImpl(ManagerFactory managerFactory, String adminPassword, boolean testMode, boolean detailsMode) {
        this.managerFactory = managerFactory;
        this.adminPassword = adminPassword;
        this.testMode = testMode;
        this.detailsMode = detailsMode;
        this.callExecutor = managerFactory.threadExecutor().getCallExecutor();
        ServerMessagesUtil.instance.getMessages();
        DataCollectorServices.init(DebugUtil.SERVER_DATA_COLLECTORS_ENABLE_PRINT_GAME_LOGS, DebugUtil.SERVER_DATA_COLLECTORS_ENABLE_SAVE_GAME_HISTORY);
        DataCollectorServices.getInstance().onServerStart();
    }

    public boolean authRegister(String sessionId, String userName, String password, String email) throws MageException {
        return this.managerFactory.sessionManager().registerUser(sessionId, userName, password, email);
    }

    private static String generateAuthToken() {
        return String.format("%06d", RANDOM.nextInt(1000000));
    }

    public boolean authSendTokenToEmail(String sessionId, String email) throws MageException {
        if (!this.managerFactory.configSettings().isAuthenticationActivated().booleanValue()) {
            this.sendErrorMessageToClient(sessionId, "Registration has been disabled on the server. You can use any name and empty password to login.");
            return false;
        }
        AuthorizedUser authorizedUser = AuthorizedUserRepository.getInstance().getByEmail(email);
        if (authorizedUser == null) {
            this.sendErrorMessageToClient(sessionId, "No user was found with the email address " + email);
            logger.info((Object)("Auth token is requested for " + email + " but there's no such user in DB"));
            return false;
        }
        String authToken = MageServerImpl.generateAuthToken();
        this.activeAuthTokens.put(email, authToken);
        String subject = "XMage Password Reset Auth Token";
        String text = "Use this auth token to reset " + authorizedUser.name + "'s password: " + authToken + '\n' + "It's valid until the next server restart.";
        boolean success = !this.managerFactory.configSettings().getMailUser().isEmpty() ? this.managerFactory.mailClient().sendMessage(email, subject, text) : this.managerFactory.mailgunClient().sendMessage(email, subject, text);
        if (!success) {
            this.sendErrorMessageToClient(sessionId, "There was an error inside the server while emailing an auth token");
            return false;
        }
        return true;
    }

    public boolean authResetPassword(String sessionId, String email, String authToken, String password) throws MageException {
        if (!this.managerFactory.configSettings().isAuthenticationActivated().booleanValue()) {
            this.sendErrorMessageToClient(sessionId, "Registration has been disabled on the server. You can use any name and empty password to login.");
            return false;
        }
        String storedAuthToken = this.activeAuthTokens.get(email);
        if (storedAuthToken == null || !storedAuthToken.equals(authToken)) {
            this.sendErrorMessageToClient(sessionId, "Invalid auth token");
            logger.info((Object)("Invalid auth token " + authToken + " is sent for " + email));
            return false;
        }
        AuthorizedUser authorizedUser = AuthorizedUserRepository.getInstance().getByEmail(email);
        if (authorizedUser == null) {
            this.sendErrorMessageToClient(sessionId, "User with that email doesn't exists");
            logger.info((Object)("Auth token is valid, but the user with email address " + email + " is no longer in the DB"));
            return false;
        }
        AuthorizedUserRepository.getInstance().remove(authorizedUser.getName());
        AuthorizedUserRepository.getInstance().add(authorizedUser.getName(), password, email);
        this.activeAuthTokens.remove(email);
        return true;
    }

    public boolean connectUser(String userName, String password, String sessionId, String restoreSessionId, MageVersion version, String userIdStr) throws MageException {
        try {
            if (version.compareTo(Main.getVersion()) != 0) {
                logger.debug((Object)("MageVersionException: userName=" + userName + ", version=" + version + " sessionId=" + sessionId));
                throw new MageVersionException(version, Main.getVersion());
            }
            return this.managerFactory.sessionManager().connectUser(sessionId, restoreSessionId, userName, password, userIdStr, this.detailsMode);
        }
        catch (MageException e) {
            if (e instanceof MageVersionException) {
                throw e;
            }
            this.handleException(e);
            return false;
        }
    }

    public boolean connectSetUserData(final String userName, final String sessionId, final UserData userData, final String clientVersion, final String userIdStr) throws MageException {
        return (Boolean)this.executeWithResult("setUserData", sessionId, (ActionWithResult)new ActionWithBooleanResult(){

            public Boolean execute() throws MageException {
                return MageServerImpl.this.managerFactory.sessionManager().setUserData(userName, sessionId, userData, clientVersion, userIdStr);
            }
        });
    }

    public boolean connectAdmin(String adminPassword, String sessionId, MageVersion version) throws MageException {
        try {
            if (version.compareTo(Main.getVersion()) != 0) {
                throw new MageException("Wrong client version " + version + ", expecting version " + Main.getVersion());
            }
            if (!adminPassword.equals(this.adminPassword)) {
                Thread.sleep(3000L);
                throw new MageException("Wrong password");
            }
            return this.managerFactory.sessionManager().connectAdmin(sessionId);
        }
        catch (Exception ex) {
            this.handleException(ex);
            return false;
        }
    }

    public TableView roomCreateTable(String sessionId, UUID roomId, MatchOptions options) throws MageException {
        return (TableView)this.executeWithResult("createTable", sessionId, (ActionWithResult)new CreateTableAction(sessionId, options, roomId));
    }

    public TableView roomCreateTournament(final String sessionId, final UUID roomId, final TournamentOptions options) throws MageException {
        return (TableView)this.executeWithResult("createTournamentTable", sessionId, (ActionWithResult)new ActionWithTableViewResult(){

            public TableView execute() throws MageException {
                try {
                    int max;
                    int aiPlayers;
                    Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId);
                    if (!session.isPresent()) {
                        logger.error((Object)("Session to found : " + sessionId));
                        return null;
                    }
                    UUID userId = session.get().getUserId();
                    Optional<User> _user = MageServerImpl.this.managerFactory.userManager().getUser(userId);
                    if (!_user.isPresent()) {
                        logger.error((Object)("User for session not found. session = " + sessionId));
                        return null;
                    }
                    User user = _user.get();
                    int notStartedTables = user.getNumberOfNotStartedTables();
                    if (notStartedTables > 1) {
                        user.showUserMessage("Create table", "You have already " + notStartedTables + " not started tables. You can't create another.");
                        throw new MageException("No message");
                    }
                    String maxAiOpponents = MageServerImpl.this.managerFactory.configSettings().getMaxAiOpponents();
                    if (maxAiOpponents != null && (aiPlayers = options.getPlayerTypes().stream().mapToInt(t -> t.isAI() && t.isWorkablePlayer() ? 1 : 0).sum()) > (max = Integer.parseInt(maxAiOpponents))) {
                        user.showUserMessage("Create tournament", "It's only allowed to use a maximum of " + max + " AI players.");
                        throw new MageException("No message");
                    }
                    int quitRatio = options.getQuitRatio();
                    if (quitRatio < user.getTourneyQuitRatio()) {
                        String message = "Your quit ratio " + user.getTourneyQuitRatio() + "% is higher than the table requirement " + quitRatio + '%';
                        user.showUserMessage("Create tournament", message);
                        throw new MageException("No message");
                    }
                    int minimumRating = options.getMinimumRating();
                    int userRating = options.getMatchOptions().isLimited() ? user.getUserData().getLimitedRating() : user.getUserData().getConstructedRating();
                    if (userRating < minimumRating) {
                        String message = "Your rating " + userRating + " is lower than the table requirement " + minimumRating;
                        user.showUserMessage("Create tournament", message);
                        throw new MageException("No message");
                    }
                    Optional<GamesRoom> room = MageServerImpl.this.managerFactory.gamesRoomManager().getRoom(roomId);
                    if (room.isPresent()) {
                        TableView table = room.get().createTournamentTable(userId, options);
                        logger.debug((Object)("Tournament table " + table.getTableId() + " created"));
                        return table;
                    }
                }
                catch (Exception ex) {
                    MageServerImpl.this.handleException(ex);
                }
                return null;
            }
        });
    }

    public void tableRemove(String sessionId, UUID roomId, UUID tableId) throws MageException {
        this.execute("removeTable", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.tableManager().removeTable(userId, tableId);
        }));
    }

    public boolean roomJoinTable(final String sessionId, final UUID roomId, final UUID tableId, final String name, final PlayerType playerType, final int skill, final DeckCardLists deckList, final String password) throws MageException {
        return (Boolean)this.executeWithResult("joinTable", sessionId, (ActionWithResult)new ActionWithBooleanResult(){

            public Boolean execute() throws MageException {
                Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId);
                if (!session.isPresent()) {
                    return false;
                }
                UUID userId = session.get().getUserId();
                logger.debug((Object)(name + " joins tableId: " + tableId));
                if (userId == null) {
                    logger.fatal((Object)("Got no userId from sessionId" + sessionId + " tableId" + tableId));
                    return false;
                }
                Optional<GamesRoom> room = MageServerImpl.this.managerFactory.gamesRoomManager().getRoom(roomId);
                if (!room.isPresent()) {
                    return false;
                }
                return room.get().joinTable(userId, tableId, name, playerType, skill, deckList, password);
            }
        });
    }

    public boolean roomJoinTournament(final String sessionId, final UUID roomId, final UUID tableId, final String name, final PlayerType playerType, final int skill, final DeckCardLists deckList, final String password) throws MageException {
        return (Boolean)this.executeWithResult("joinTournamentTable", sessionId, (ActionWithResult)new ActionWithBooleanResult(){

            public Boolean execute() throws MageException {
                Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId);
                if (!session.isPresent()) {
                    return false;
                }
                UUID userId = session.get().getUserId();
                if (logger.isTraceEnabled()) {
                    Optional<User> user = MageServerImpl.this.managerFactory.userManager().getUser(userId);
                    user.ifPresent(user1 -> logger.trace((Object)("join tourney tableId: " + tableId + ' ' + name)));
                }
                if (userId == null) {
                    logger.fatal((Object)("Got no userId from sessionId" + sessionId + " tableId" + tableId));
                    return false;
                }
                Optional<GamesRoom> room = MageServerImpl.this.managerFactory.gamesRoomManager().getRoom(roomId);
                if (room.isPresent()) {
                    return room.get().joinTournamentTable(userId, tableId, name, playerType, skill, deckList, password);
                }
                return null;
            }
        });
    }

    public boolean deckSubmit(final String sessionId, final UUID tableId, final DeckCardLists deckList) throws MageException {
        return (Boolean)this.executeWithResult("deckSubmit", sessionId, (ActionWithResult)new ActionWithBooleanResult(){

            public Boolean execute() throws MageException {
                Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId);
                if (!session.isPresent()) {
                    return false;
                }
                UUID userId = session.get().getUserId();
                boolean ret = MageServerImpl.this.managerFactory.tableManager().submitDeck(userId, tableId, deckList);
                logger.debug((Object)("Session " + sessionId + " submitted deck"));
                return ret;
            }
        });
    }

    public void deckSave(String sessionId, UUID tableId, DeckCardLists deckList) throws MageException {
        this.execute("updateDeck", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.tableManager().updateDeck(userId, tableId, deckList);
                logger.trace((Object)("Session " + sessionId + " updated deck"));
            }
        });
    }

    public List<TableView> roomGetAllTables(UUID roomId) throws MageException {
        try {
            Optional<GamesRoom> room = this.managerFactory.gamesRoomManager().getRoom(roomId);
            if (room.isPresent()) {
                return room.get().getTables();
            }
            return new ArrayList<TableView>();
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public List<MatchView> roomGetFinishedMatches(UUID roomId) throws MageException {
        try {
            return this.managerFactory.gamesRoomManager().getRoom(roomId).map(GamesRoom::getFinished).orElse(new ArrayList());
        }
        catch (Exception ex) {
            this.handleException(ex);
            return new ArrayList<MatchView>();
        }
    }

    public List<RoomUsersView> roomGetUsers(UUID roomId) throws MageException {
        try {
            Optional<GamesRoom> room = this.managerFactory.gamesRoomManager().getRoom(roomId);
            if (room.isPresent()) {
                return room.get().getRoomUsersInfo();
            }
            return new ArrayList<RoomUsersView>();
        }
        catch (Exception ex) {
            this.handleException(ex);
            return new ArrayList<RoomUsersView>();
        }
    }

    public TableView roomGetTableById(UUID roomId, UUID tableId) throws MageException {
        try {
            Optional<GamesRoom> room = this.managerFactory.gamesRoomManager().getRoom(roomId);
            return room.flatMap(r -> r.getTable(tableId)).orElse(null);
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public boolean ping(String sessionId, String pingInfo) {
        return this.managerFactory.sessionManager().extendUserSession(sessionId, pingInfo);
    }

    public boolean matchStart(String sessionId, UUID roomId, UUID tableId) throws MageException {
        Optional<TableController> controller = this.managerFactory.tableManager().getController(tableId);
        if (!controller.isPresent()) {
            logger.error((Object)("table not found : " + tableId));
            return false;
        }
        if (!controller.get().changeTableStateToStarting()) {
            return false;
        }
        this.execute("startMatch", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.tableManager().startMatch(userId, roomId, tableId);
            }
        });
        return true;
    }

    public boolean tournamentStart(String sessionId, UUID roomId, UUID tableId) throws MageException {
        Optional<TableController> controller = this.managerFactory.tableManager().getController(tableId);
        if (!controller.isPresent()) {
            logger.error((Object)("table not found : " + tableId));
            return false;
        }
        if (!controller.get().changeTableStateToStarting()) {
            return false;
        }
        this.execute("startTournament", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.tableManager().startTournament(userId, roomId, tableId);
            }
        });
        return true;
    }

    public TournamentView tournamentFindById(UUID tournamentId) throws MageException {
        try {
            return this.managerFactory.tournamentManager().getTournamentView(tournamentId);
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public void chatSendMessage(UUID chatId, String userName, String message) throws MageException {
        if (message.length() > 500) {
            logger.error((Object)("Chat message too big: " + message.length() + ", from user " + userName));
            return;
        }
        try {
            this.callExecutor.execute(() -> this.managerFactory.chatManager().broadcast(chatId, userName, HtmlEscape.escapeHtml4((String)message), ChatMessage.MessageColor.BLUE, true, null, ChatMessage.MessageType.TALK, null));
        }
        catch (Exception ex) {
            this.handleException(ex);
        }
    }

    public void chatJoin(UUID chatId, String sessionId, String userName) throws MageException {
        this.execute("joinChat", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.chatManager().joinChat(chatId, userId);
        }));
    }

    public void chatLeave(UUID chatId, String sessionId) throws MageException {
        this.execute("leaveChat", sessionId, () -> {
            if (chatId != null) {
                this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
                    UUID userId = session.getUserId();
                    this.managerFactory.chatManager().leaveChat(chatId, userId);
                });
            }
        });
    }

    public UUID serverGetMainRoomId() throws MageException {
        try {
            return this.managerFactory.gamesRoomManager().getMainRoomId();
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public UUID chatFindByRoom(UUID roomId) throws MageException {
        try {
            Optional<GamesRoom> room = this.managerFactory.gamesRoomManager().getRoom(roomId);
            if (!room.isPresent()) {
                logger.error((Object)("roomId not found : " + roomId));
                return null;
            }
            return room.get().getChatId();
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public boolean tableIsOwner(final String sessionId, UUID roomId, final UUID tableId) throws MageException {
        return (Boolean)this.executeWithResult("isTableOwner", sessionId, (ActionWithResult)new ActionWithBooleanResult(){

            public Boolean execute() {
                Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId);
                if (!session.isPresent()) {
                    return false;
                }
                UUID userId = session.get().getUserId();
                return MageServerImpl.this.managerFactory.tableManager().isTableOwner(tableId, userId);
            }
        });
    }

    public void tableSwapSeats(String sessionId, UUID roomId, UUID tableId, int seatNum1, int seatNum2) throws MageException {
        this.execute("swapSeats", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.tableManager().swapSeats(tableId, userId, seatNum1, seatNum2);
        }));
    }

    public boolean roomLeaveTableOrTournament(String sessionId, UUID roomId, UUID tableId) throws MageException {
        Optional<TableController> tableController = this.managerFactory.tableManager().getController(tableId);
        if (tableController.isPresent()) {
            TableState tableState = tableController.get().getTableState();
            if (tableState != TableState.WAITING && tableState != TableState.READY_TO_START) {
                return false;
            }
            this.execute("leaveTable", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
                UUID userId = session.getUserId();
                this.managerFactory.gamesRoomManager().getRoom(roomId).ifPresent(room -> room.leaveTable(userId, tableId));
            }));
        } else {
            logger.trace((Object)("table not found : " + tableId));
        }
        return true;
    }

    public UUID chatFindByTable(UUID tableId) throws MageException {
        try {
            return this.managerFactory.tableManager().getChatId(tableId).orElse(null);
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public void gameJoin(UUID gameId, String sessionId) throws MageException {
        this.execute("joinGame", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.gameManager().joinGame(gameId, userId);
        }));
    }

    public void draftJoin(UUID draftId, String sessionId) throws MageException {
        this.execute("joinDraft", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.draftManager().joinDraft(draftId, userId);
        }));
    }

    public void tournamentJoin(UUID tournamentId, String sessionId) throws MageException {
        this.execute("joinTournament", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.tournamentManager().joinTournament(tournamentId, userId);
            }
        });
    }

    public UUID chatFindByGame(UUID gameId) throws MageException {
        try {
            return this.managerFactory.gameManager().getChatId(gameId).orElse(null);
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public UUID chatFindByTournament(UUID tournamentId) throws MageException {
        try {
            return this.managerFactory.tournamentManager().getChatId(tournamentId).orElse(null);
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public void sendPlayerUUID(UUID gameId, String sessionId, UUID data) throws MageException {
        this.execute("sendPlayerUUID", sessionId, () -> {
            Optional<User> user = this.managerFactory.sessionManager().getUser(sessionId);
            if (user.isPresent()) {
                user.get().sendPlayerUUID(gameId, data);
            } else {
                logger.warn((Object)("Your session expired: gameId=" + gameId + ", sessionId=" + sessionId));
            }
        });
    }

    public void sendPlayerString(UUID gameId, String sessionId, String data) throws MageException {
        this.execute("sendPlayerString", sessionId, () -> {
            Optional<User> user = this.managerFactory.sessionManager().getUser(sessionId);
            if (user.isPresent()) {
                user.get().sendPlayerString(gameId, data);
            } else {
                logger.warn((Object)("Your session expired: gameId=" + gameId + ", sessionId=" + sessionId));
            }
        });
    }

    public void sendPlayerManaType(UUID gameId, UUID playerId, String sessionId, ManaType data) throws MageException {
        this.execute("sendPlayerManaType", sessionId, () -> {
            Optional<User> user = this.managerFactory.sessionManager().getUser(sessionId);
            if (user.isPresent()) {
                user.get().sendPlayerManaType(gameId, playerId, data);
            } else {
                logger.warn((Object)("Your session expired: gameId=" + gameId + ", sessionId=" + sessionId));
            }
        });
    }

    public void sendPlayerBoolean(UUID gameId, String sessionId, Boolean data) throws MageException {
        this.execute("sendPlayerBoolean", sessionId, () -> {
            Optional<User> user = this.managerFactory.sessionManager().getUser(sessionId);
            if (user.isPresent()) {
                user.get().sendPlayerBoolean(gameId, data);
            } else {
                logger.warn((Object)("Your session expired: gameId=" + gameId + ", sessionId=" + sessionId));
            }
        });
    }

    public void sendPlayerInteger(UUID gameId, String sessionId, Integer data) throws MageException {
        this.execute("sendPlayerInteger", sessionId, () -> {
            Optional<User> user = this.managerFactory.sessionManager().getUser(sessionId);
            if (user.isPresent()) {
                user.get().sendPlayerInteger(gameId, data);
            } else {
                logger.warn((Object)("Your session expired: gameId=" + gameId + ", sessionId=" + sessionId));
            }
        });
    }

    public DraftPickView sendDraftCardPick(UUID draftId, String sessionId, UUID cardPick, Set<UUID> hiddenCards) throws MageException {
        return (DraftPickView)this.executeWithResult("sendCardPick", sessionId, (ActionWithResult)new SendCardPickAction(sessionId, draftId, cardPick, hiddenCards));
    }

    public void sendDraftCardMark(UUID draftId, String sessionId, UUID cardPick) throws MageException {
        this.execute("sendCardMark", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.draftManager().sendCardMark(draftId, userId, cardPick);
        }));
    }

    public void draftSetBoosterLoaded(UUID draftId, String sessionId) throws MageException {
        this.execute("setBoosterLoaded", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.draftManager().setBoosterLoaded(draftId, userId);
        }));
    }

    public void matchQuit(UUID gameId, String sessionId) throws MageException {
        this.execute("quitMatch", sessionId, () -> {
            try {
                this.callExecutor.execute(() -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
                    UUID userId = session.getUserId();
                    this.managerFactory.gameManager().quitMatch(gameId, userId);
                }));
            }
            catch (Exception ex) {
                this.handleException(ex);
            }
        });
    }

    public void tournamentQuit(UUID tournamentId, String sessionId) throws MageException {
        this.execute("quitTournament", sessionId, () -> {
            try {
                this.callExecutor.execute(() -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
                    UUID userId = session.getUserId();
                    this.managerFactory.tournamentManager().quit(tournamentId, userId);
                }));
            }
            catch (Exception ex) {
                this.handleException(ex);
            }
        });
    }

    public void draftQuit(UUID draftId, String sessionId) throws MageException {
        this.execute("quitDraft", sessionId, () -> {
            try {
                this.callExecutor.execute(() -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
                    UUID userId = session.getUserId();
                    UUID tableId = this.managerFactory.draftManager().getControllerByDraftId(draftId).getTableId();
                    Table table = this.managerFactory.tableManager().getTable(tableId);
                    if (table.isTournament()) {
                        UUID tournamentId = table.getTournament().getId();
                        this.managerFactory.tournamentManager().quit(tournamentId, userId);
                    }
                }));
            }
            catch (Exception ex) {
                this.handleException(ex);
            }
        });
    }

    public void sendPlayerAction(PlayerAction playerAction, UUID gameId, String sessionId, Object data) throws MageException {
        this.execute("sendPlayerAction", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.gameManager().sendPlayerAction(playerAction, gameId, userId, data);
        }));
    }

    public boolean roomWatchTable(final String sessionId, final UUID roomId, final UUID tableId) throws MageException {
        return (Boolean)this.executeWithResult("setUserData", sessionId, (ActionWithResult)new ActionWithBooleanResult(){

            public Boolean execute() throws MageException {
                Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId);
                if (!session.isPresent()) {
                    logger.error((Object)("Session not found : " + sessionId));
                    return false;
                }
                UUID userId = session.get().getUserId();
                if (MageServerImpl.this.managerFactory.gamesRoomManager().getRoom(roomId).isPresent()) {
                    return MageServerImpl.this.managerFactory.gamesRoomManager().getRoom(roomId).get().watchTable(userId, tableId);
                }
                return false;
            }
        });
    }

    public boolean roomWatchTournament(String sessionId, UUID tableId) throws MageException {
        return (Boolean)this.executeWithResult("setUserData", sessionId, (ActionWithResult)new WatchTournamentTableAction(sessionId, tableId));
    }

    public boolean gameWatchStart(final UUID gameId, final String sessionId) throws MageException {
        return this.executeWithResult("watchGame", sessionId, new ActionWithResult<Boolean>(){

            public Boolean execute() throws MageException {
                return MageServerImpl.this.managerFactory.sessionManager().getSession(sessionId).map(session -> {
                    UUID userId = session.getUserId();
                    return MageServerImpl.this.managerFactory.gameManager().watchGame(gameId, userId);
                }).orElse(false);
            }

            public Boolean negativeResult() {
                return false;
            }
        });
    }

    public void gameWatchStop(UUID gameId, String sessionId) throws MageException {
        this.execute("stopWatching", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.userManager().getUser(userId).ifPresent(user -> {
                this.managerFactory.gameManager().stopWatching(gameId, userId);
                user.removeGameWatchInfo(gameId);
            });
        }));
    }

    public void replayInit(UUID gameId, String sessionId) throws MageException {
        this.execute("replayGame", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.replayManager().replayGame(gameId, userId);
        }));
    }

    public void replayStart(UUID gameId, String sessionId) throws MageException {
        this.execute("startReplay", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.replayManager().startReplay(gameId, userId);
        }));
    }

    public void replayStop(UUID gameId, String sessionId) throws MageException {
        this.execute("stopReplay", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.replayManager().stopReplay(gameId, userId);
            }
        });
    }

    public void replayNext(UUID gameId, String sessionId) throws MageException {
        this.execute("nextPlay", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.replayManager().nextPlay(gameId, userId);
            }
        });
    }

    public void replayPrevious(UUID gameId, String sessionId) throws MageException {
        this.execute("previousPlay", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.replayManager().previousPlay(gameId, userId);
            }
        });
    }

    public void replaySkipForward(UUID gameId, String sessionId, int moves) throws MageException {
        this.execute("skipForward", sessionId, () -> {
            Optional<Session> session = this.managerFactory.sessionManager().getSession(sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + sessionId));
            } else {
                UUID userId = session.get().getUserId();
                this.managerFactory.replayManager().skipForward(gameId, userId, moves);
            }
        });
    }

    public ServerState getServerState() throws MageException {
        try {
            Thread.sleep(1000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        try {
            return new ServerState(GameFactory.instance.getGameTypes(), TournamentFactory.instance.getTournamentTypes(), PlayerFactory.instance.getPlayerTypes().toArray(new PlayerType[PlayerFactory.instance.getPlayerTypes().size()]), DeckValidatorFactory.instance.getDeckTypes().toArray(new String[DeckValidatorFactory.instance.getDeckTypes().size()]), CubeFactory.instance.getDraftCubes().toArray(new String[CubeFactory.instance.getDraftCubes().size()]), this.testMode, Main.getVersion(), CardRepository.instance.getContentVersionConstant(), ExpansionRepository.instance.getContentVersionConstant());
        }
        catch (Exception ex) {
            this.handleException(ex);
            return null;
        }
    }

    public void cheatShow(UUID gameId, String sessionId, UUID playerId) throws MageException {
        this.execute("cheatShow", sessionId, () -> {
            if (this.testMode) {
                this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
                    UUID userId = session.getUserId();
                    this.managerFactory.gameManager().cheatShow(gameId, userId, playerId);
                });
            }
        });
    }

    public void handleException(Exception ex) throws MageException {
        if (ex.getMessage() != null && !ex.getMessage().equals("No message")) {
            throw new MageException(ex.getMessage());
        }
        if (ex instanceof ConcurrentModificationException) {
            logger.error((Object)"wrong threads sync error", (Throwable)ex);
        } else {
            logger.error((Object)"unknown error", (Throwable)ex);
        }
    }

    public GameView gameGetView(UUID gameId, String sessionId, UUID playerId) throws MageException {
        return (GameView)this.executeWithResult("getGameView", sessionId, (ActionWithResult)new GetGameViewAction(sessionId, gameId, playerId));
    }

    public List<UserView> adminGetUsers(String sessionId) throws MageException {
        return (List)this.executeWithResult("adminGetUsers", sessionId, (ActionWithResult)new GetUsersAction(), true);
    }

    public void adminDisconnectUser(String sessionId, String userSessionId) throws MageException {
        this.execute("adminDisconnectUser", sessionId, () -> this.managerFactory.sessionManager().disconnectAnother(sessionId, userSessionId), true);
    }

    public void adminMuteUser(String sessionId, String userName, long durationMinutes) throws MageException {
        this.execute("adminMuteUser", sessionId, () -> this.managerFactory.userManager().getUserByName(userName).ifPresent(user -> {
            Date muteUntil = new Date(Calendar.getInstance().getTimeInMillis() + durationMinutes * 60000L);
            user.showUserMessage("Admin info", "You were muted for chat messages until " + SystemUtil.dateFormat.format(muteUntil) + '.');
            user.setChatLockedUntil(muteUntil);
        }), true);
    }

    public void adminLockUser(String sessionId, String userName, long durationMinutes) throws MageException {
        this.execute("adminLockUser", sessionId, () -> this.managerFactory.userManager().getUserByName(userName).ifPresent(user -> {
            Date lockUntil = new Date(Calendar.getInstance().getTimeInMillis() + durationMinutes * 60000L);
            user.showUserMessage("Admin info", "Your user profile was locked until " + SystemUtil.dateFormat.format(lockUntil) + '.');
            user.setLockedUntil(lockUntil);
            if (user.isConnected()) {
                this.managerFactory.sessionManager().disconnectAnother(sessionId, user.getSessionId());
            }
        }), true);
    }

    public void adminActivateUser(String sessionId, String userName, boolean active) throws MageException {
        this.execute("adminActivateUser", sessionId, () -> {
            AuthorizedUser authorizedUser = AuthorizedUserRepository.getInstance().getByName(userName);
            Optional<User> u = this.managerFactory.userManager().getUserByName(userName);
            if (u.isPresent()) {
                User user = u.get();
                user.setActive(active);
                if (!user.isActive() && user.isConnected()) {
                    this.managerFactory.sessionManager().disconnectAnother(sessionId, user.getSessionId());
                }
            } else if (authorizedUser != null) {
                User theUser = new User(this.managerFactory, userName, "localhost", authorizedUser);
                theUser.setActive(active);
            }
        }, true);
    }

    public void adminToggleActivateUser(String sessionId, String userName) throws MageException {
        this.execute("adminToggleActivateUser", sessionId, () -> this.managerFactory.userManager().getUserByName(userName).ifPresent(user -> {
            user.setActive(!user.isActive());
            if (!user.isActive() && user.isConnected()) {
                this.managerFactory.sessionManager().disconnectAnother(sessionId, user.getSessionId());
            }
        }), true);
    }

    public void adminEndUserSession(String sessionId, String userSessionId) throws MageException {
        this.execute("adminEndUserSession", sessionId, () -> this.managerFactory.sessionManager().disconnectAnother(sessionId, userSessionId), true);
    }

    public void adminTableRemove(String sessionId, UUID tableId) throws MageException {
        this.execute("adminTableRemove", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> {
            UUID userId = session.getUserId();
            this.managerFactory.tableManager().removeTable(userId, tableId);
        }), true);
    }

    public Object serverGetPromotionMessages(String sessionId) throws MageException {
        return this.executeWithResult("serverGetPromotionMessages", sessionId, (ActionWithResult)new GetPromotionMessagesAction());
    }

    public void serverAddFeedbackMessage(String sessionId, String username, String title, String type, String message, String email) throws MageException {
        if (title != null && message != null) {
            this.execute("sendFeedbackMessage", sessionId, () -> this.managerFactory.sessionManager().getSession(sessionId).ifPresent(session -> FeedbackServiceImpl.instance.feedback(username, title, type, message, email, session.getHost())));
        }
    }

    public void adminSendBroadcastMessage(String sessionId, String message) throws MageException {
        if (message != null) {
            this.execute("sendBroadcastMessage", sessionId, () -> {
                for (User user : this.managerFactory.userManager().getUsers()) {
                    if (message.toLowerCase(Locale.ENGLISH).startsWith("warn")) {
                        user.fireCallback(new ClientCallback(ClientCallbackMethod.SERVER_MESSAGE, null, (Object)new ChatMessage("SERVER", message, null, null, ChatMessage.MessageColor.RED)));
                        continue;
                    }
                    user.fireCallback(new ClientCallback(ClientCallbackMethod.SERVER_MESSAGE, null, (Object)new ChatMessage("SERVER", message, null, null, ChatMessage.MessageColor.BLUE)));
                }
            }, true);
        }
    }

    private void sendErrorMessageToClient(String sessionId, String message) throws MageException {
        this.execute("sendErrorMessageToClient", sessionId, () -> this.managerFactory.sessionManager().sendErrorMessageToClient(sessionId, message));
    }

    protected void execute(String actionName, String sessionId, Action action, boolean checkAdminRights) throws MageException {
        if (checkAdminRights && !this.managerFactory.sessionManager().checkAdminAccess(sessionId)) {
            return;
        }
        this.execute(actionName, sessionId, action);
    }

    protected void execute(String actionName, String sessionId, Action action) throws MageException {
        if (this.managerFactory.sessionManager().isValidSession(sessionId)) {
            try {
                this.callExecutor.execute(() -> {
                    if (this.managerFactory.sessionManager().isValidSession(sessionId)) {
                        try {
                            action.execute();
                        }
                        catch (MageException me) {
                            throw new RuntimeException(me);
                        }
                    }
                });
            }
            catch (Exception ex) {
                this.handleException(ex);
            }
        }
    }

    protected <T> T executeWithResult(String actionName, String sessionId, ActionWithResult<T> action, boolean checkAdminRights) throws MageException {
        if (checkAdminRights && !this.managerFactory.sessionManager().checkAdminAccess(sessionId)) {
            return (T)action.negativeResult();
        }
        return this.executeWithResult(actionName, sessionId, action);
    }

    protected <T> T executeWithResult(String actionName, String sessionId, ActionWithResult<T> action) throws MageException {
        if (this.managerFactory.sessionManager().isValidSession(sessionId)) {
            try {
                return (T)action.execute();
            }
            catch (Exception ex) {
                this.handleException(ex);
            }
        }
        return (T)action.negativeResult();
    }

    private class CreateTableAction
    extends ActionWithTableViewResult {
        private final String sessionId;
        private final MatchOptions options;
        private final UUID roomId;

        public CreateTableAction(String sessionId, MatchOptions options, UUID roomId) {
            this.sessionId = sessionId;
            this.options = options;
            this.roomId = roomId;
        }

        public TableView execute() throws MageException {
            Session session = MageServerImpl.this.managerFactory.sessionManager().getSession(this.sessionId).orElse(null);
            if (session == null) {
                return null;
            }
            UUID userId = session.getUserId();
            User user = MageServerImpl.this.managerFactory.userManager().getUser(userId).orElse(null);
            if (user == null) {
                return null;
            }
            int notStartedTables = user.getNumberOfNotStartedTables();
            if (notStartedTables > 1) {
                user.showUserMessage("Create table", "You have already " + notStartedTables + " not started tables. You can't create another.");
                throw new MageException("User " + user.getName() + " can't create table: too much started");
            }
            int quitRatio = this.options.getQuitRatio();
            if (quitRatio < user.getMatchQuitRatio()) {
                user.showUserMessage("Create table", "Your quit ratio " + user.getMatchQuitRatio() + "% is higher than the table requirement " + quitRatio + '%');
                throw new MageException("User " + user.getName() + " can't create table: incompatible quit ratio");
            }
            int minimumRating = this.options.getMinimumRating();
            int userRating = this.options.isLimited() ? user.getUserData().getLimitedRating() : user.getUserData().getConstructedRating();
            if (userRating < minimumRating) {
                String message = "Your rating " + userRating + " is lower than the table requirement " + minimumRating;
                user.showUserMessage("Create table", message);
                throw new MageException("User " + user.getName() + " can't create table: incompatible rating");
            }
            GamesRoom room = MageServerImpl.this.managerFactory.gamesRoomManager().getRoom(this.roomId).orElse(null);
            if (room == null) {
                return null;
            }
            return room.createTable(userId, this.options);
        }
    }

    private class SendCardPickAction
    extends ActionWithNullNegativeResult<DraftPickView> {
        private final String sessionId;
        private final UUID draftId;
        private final UUID cardPick;
        private final Set<UUID> hiddenCards;

        public SendCardPickAction(String sessionId, UUID draftId, UUID cardPick, Set<UUID> hiddenCards) {
            this.sessionId = sessionId;
            this.draftId = draftId;
            this.cardPick = cardPick;
            this.hiddenCards = hiddenCards;
        }

        public DraftPickView execute() {
            Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(this.sessionId);
            if (session.isPresent()) {
                return MageServerImpl.this.managerFactory.draftManager().sendCardPick(this.draftId, session.get().getUserId(), this.cardPick, this.hiddenCards);
            }
            logger.error((Object)("Session not found sessionId: " + this.sessionId + "  draftId:" + this.draftId));
            return null;
        }
    }

    private class WatchTournamentTableAction
    extends ActionWithBooleanResult {
        private final String sessionId;
        private final UUID tableId;

        public WatchTournamentTableAction(String sessionId, UUID tableId) {
            this.sessionId = sessionId;
            this.tableId = tableId;
        }

        public Boolean execute() throws MageException {
            Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(this.sessionId);
            if (!session.isPresent()) {
                return false;
            }
            UUID userId = session.get().getUserId();
            return MageServerImpl.this.managerFactory.tableManager().watchTable(userId, this.tableId);
        }
    }

    private class GetGameViewAction
    extends ActionWithNullNegativeResult<GameView> {
        private final String sessionId;
        private final UUID gameId;
        private final UUID playerId;

        public GetGameViewAction(String sessionId, UUID gameId, UUID playerId) {
            this.sessionId = sessionId;
            this.gameId = gameId;
            this.playerId = playerId;
        }

        public GameView execute() throws MageException {
            Optional<Session> session = MageServerImpl.this.managerFactory.sessionManager().getSession(this.sessionId);
            if (!session.isPresent()) {
                logger.error((Object)("Session not found : " + this.sessionId));
                return null;
            }
            return MageServerImpl.this.managerFactory.gameManager().getGameView(this.gameId, this.playerId);
        }
    }

    private class GetUsersAction
    extends ActionWithNullNegativeResult<List<UserView>> {
        private GetUsersAction() {
        }

        public List<UserView> execute() throws MageException {
            return MageServerImpl.this.managerFactory.userManager().getUserInfoList();
        }
    }

    private static class GetPromotionMessagesAction
    extends ActionWithNullNegativeResult<Object> {
        private GetPromotionMessagesAction() {
        }

        public Object execute() throws MageException {
            return CompressUtil.compress(ServerMessagesUtil.instance.getMessages());
        }
    }
}

